/**
 * jobqueue.c
 *
 * Дана програма моделює роботу багатопотокового паралельного сервера без
 * зворотного зв'язку між серверними і клієнтськими потоками (але якщо
 * клієнтські потоки представляють зовнішніх клієнтів, може мати місце 
 * зворотний зв'язок між серверними потоками і зовнішніми клієнтами;
 * для цього клієнтські потоки в структуру, яка описує завдання для 
 * серверного потоку, повинні включати інформацію про спосіб зв'язку з
 * зовнішнім клієнтом). Використовується фіксована кількість завчасно
 * створених серверних потоків. Кілька клієнтських потоків (які, можливо,
 * представляють зовнішніх клієнтів) періодично виробляють завдання (jobs)
 * для серверних потоків, які розміщують у черзі завдань (jobqueue).
 * Серверні потоки витягують завдання з черги завдань і обробляють їх.
 * Клієнтські потоки, помістивши завдання в чергу, продовжують роботу
 * (наприклад, чекають запиту від нового зовнішнього клієнта). Програма
 * виводить повідомлення про розміщення в черзі чергового завдання і
 * результат обробки кожного завдання. Черга завдань реалізована як
 * зв'язаний список структур. Доступ до неї регулюється за допомогою
 * м'ютекса (job_queue_mutex) і семафора (job_queue_count). 
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Кількість серверних (NSERVERS) і клієнтських (NCLIENTS) потоків */
enum { NSERVERS = 5, NCLIENTS = 3 };


/* Структура, яка представляє завдання */
struct job {
        int client_id;
        /* ... */
        struct job *next;
};

/* Покажчики на початок та кінець черги завдань. */
struct job *job_queue_head, *job_queue_tail;

pthread_mutex_t job_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
sem_t job_queue_count;

int init_job_queue();
void *client(void *), *server(void *);


int main(int argc, char **argv)
{
        pthread_t client_threads[NCLIENTS];
        pthread_t server_threads[NSERVERS];
        int client_args[NCLIENTS];
        int server_args[NSERVERS];
        int i, rval;

        /* Виконує ініціалізацію черги завдань. */
        if (init_job_queue() != 0)
                exit(EXIT_FAILURE);
        /* Створює серверні потоки. */
        for (i = 0; i < NSERVERS; i++) {
                server_args[i] = i;
                rval = pthread_create(&server_threads[i], NULL, server, &server_args[i]);
                if (rval != 0) {
                        fprintf(stderr, "Error creating thread: %s\n", strerror(rval));
                        exit(EXIT_FAILURE);
                }
        }
        /* Створює клієнтські потоки (які виступають як представники
           зовнішніх клієнтів, можливо). */
        for (i = 0; i < NCLIENTS; i++) {
                client_args[i] = i;
                rval = pthread_create(&client_threads[i], NULL, client,
                                                        &client_args[i]);
                if (rval != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                        exit(EXIT_FAILURE);
                }
        }
        /* Чекає завершення клієнтських потоків. */
        for (i = 0; i < NCLIENTS; i++) {
                rval = pthread_join(client_threads[i], NULL);
                assert(rval == 0);
        }
        /* Чекає завершення серверних потоків. */
        for (i = 0; i < NSERVERS; i++) {
                rval = pthread_join(server_threads[i], NULL);
                assert(rval == 0);
        }

        exit(EXIT_SUCCESS);
}

/**
 * Розміщує структуру для нового завдання і виконує її ініціалізацію.
 * Повертає: покажчик на створену структуру у випадку успіху, NULL -
 * у випадку невдачі.
 */
struct job *create_job(int client_id)
{
        struct job *new_job;

        new_job = malloc(sizeof(*new_job));
        if (new_job == NULL) {
                fprintf(stderr, "Not enough memory to create new job\n");
                return NULL;
        }
        new_job->client_id = client_id;
        /* ... */
        new_job->next = NULL;
        return new_job;
}

/**
 * Звільняє пам'ять, виділену для завдання.
 */
void free_job(struct job *job)
{
        assert(job != NULL);
        free(job);
}

/**
 * Виконує ініціалізацію черги завдань.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int init_job_queue()
{
        job_queue_head = NULL;
        job_queue_tail = NULL;
        if (sem_init(&job_queue_count, 0, 0) == -1) {
	       fprintf(stderr, "Error initializing semaphore: %s\n",
	                                               strerror(errno));
	       return 1;
        }
        return 0;
}

/**
 * Додає завдання до (хвоста) черги завдань.
 * Аргументи: new_job - покажчик на структуру із новим завданням.
 */
void enqueue_job(struct job *new_job)
{
        int rval;

        assert(new_job != NULL);

        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        if (job_queue_tail == NULL) {
	       assert(job_queue_head == NULL);
	       job_queue_head = new_job;
        } else
	       job_queue_tail->next = new_job;
        job_queue_tail = new_job;

        printf("A job was requested by client %d\n", new_job->client_id);

        rval = sem_post(&job_queue_count);
        assert(rval == 0);

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return;
}

/**
 * Вилучає завдання з (голови) черги завдань.
 * Повертає покажчик на структуру з вилученим із черги завданням.
 */
struct job* dequeue_job()
{
        struct job *next_job;
        int rval;

        /* Якщо черга порожня, чекає, доки в ній з'явиться завдання. */
        rval = sem_wait(&job_queue_count);
        assert(rval == 0);

        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        next_job = job_queue_head;
        job_queue_head = job_queue_head->next;
        if (job_queue_head == NULL)
	       job_queue_tail = NULL;

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return next_job;
}

/**
 * Головна функція клієнтського потоку.
 * Аргументи: arg - покажчик на змінну типу int, яка зберігає номер
 * клієнтського потоку.
 */
void *client(void *arg)
{
        int number;             /* Номер клієнтського потоку */

        number = *((int *) arg);
        for (;;) {
                struct job *new_job;

                /* Виконує якусь роботу (можливо, взаємодіє з зовнішнім
                   клієнтом)... */

                /* Створює нове завдання. */
                new_job = create_job(number);
                if (new_job == NULL)
                        continue;
                /* Додає завдання до черги. */
                enqueue_job(new_job);

                /* Виконує якусь іншу роботу... */

                /* Це тут тільки для того, щоб уповільнити роботу
                   клієнтських потоків. */
                sleep(1); 
        }
        return NULL;
}

/** 
 * Виконує обробку завдання.
 * Аргументи: job - покажчик на структуру з завданням, яке треба обробити.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int process_job(struct job *job)
{
        assert(job != NULL);
        /* ... */
        return 0;
}

/**
 * Головна функція серверного потоку.
 * Аргументи: arg - покажчик на змінну типу int, яка зберігає номер
 * серверного потоку.
 */
void *server(void *arg)
{
        int number;             /* Номер серверного потоку */

        number = *((int *) arg);
        for (;;) {
                struct job *next_job;

                /* Вилучає з черги нове завдання. */
                next_job = dequeue_job();
                /* Обробляє завдання. */
                if (process_job(next_job) != 0)
                        fprintf(stderr, "Error processing job\n");
                else
                        printf("A job from client %d was processed by"
                                " server %d\n", next_job->client_id,
                                                                number);
                /* Звільняє пам'ять, виділену для завдання. */
                free_job(next_job);
        }
        return NULL;
}
